<?php
session_start();
include 'conexao.php';

// Redirecionar se o usuário não for um professor
$perfil = $_SESSION['usuario']['perfil'] ?? '';
if (strtoupper(trim($perfil)) !== 'PROFESSOR') {
    header('Location: login.php');
    exit;
}

// Verificar se a requisição é POST e se os dados necessários estão presentes
if ($_SERVER['REQUEST_METHOD'] !== 'POST' || !isset($_POST['aluno_id'], $_POST['disciplina_id'], $_POST['turma_id'], $_POST['trimestre'])) {
    die("Erro: Dados do formulário incompletos ou requisição inválida.");
}

$disciplina_id = (int)$_POST['disciplina_id'];
$turma_id = (int)$_POST['turma_id'];
$trimestre_selecionado = (int)$_POST['trimestre'];
$ano_lectivo = '2025/2026'; // Certifique-se de que o ano letivo está correto

$conn->begin_transaction();

try {
    // 1. Preparar a consulta para obter as notas existentes de um aluno para TODOS os trimestres
    // Isso é crucial para manter os dados dos trimestres não-selecionados
    $sql_busca_todas = "SELECT * FROM notas WHERE aluno_id = ? AND disciplina_id = ? AND turma_id = ? AND ano_lectivo = ?";
    $stmt_busca_todas = $conn->prepare($sql_busca_todas);

    // 2. Preparar a consulta para a atualização (UPSERT)
    // ATENÇÃO: Colunas 'av1', 'av2', 'av3' foram removidas e 'mac', 'npp' adicionadas.
    // Certifique-se de que sua tabela 'notas' no banco de dados reflita isso.
    $sql_upsert = "
        REPLACE INTO notas (
            aluno_id, turma_id, disciplina_id, ano_lectivo, trimestre, classe,
            mac, npp, npt,
            mt1, mt2, mt3,
            mfd, mf,
            neo, nee, mec, exame, nota_final
        ) VALUES (
            ?, ?, ?, ?, ?, ?,
            ?, ?, ?,
            ?, ?, ?,
            ?, ?,
            ?, ?, ?, ?, ?
        )
    ";
    $stmt_upsert = $conn->prepare($sql_upsert);

    if (!$stmt_upsert) {
        throw new Exception("Erro na preparação do statement: " . $conn->error);
    }
    
    // Obter a classe da turma uma vez
    $classe = null;
    $sql_classe = "SELECT classe FROM turmas WHERE id = ?";
    $stmt_classe = $conn->prepare($sql_classe);
    $stmt_classe->bind_param("i", $turma_id);
    $stmt_classe->execute();
    $res_classe = $stmt_classe->get_result();
    if ($res_classe->num_rows > 0) {
        $classe = $res_classe->fetch_assoc()['classe'];
    }
    $stmt_classe->close();

    foreach ($_POST['aluno_id'] as $aluno_id) {
        $aluno_id = (int)$aluno_id;

        // Array para armazenar as notas que serão salvas para ESTE aluno, por trimestre
        // Inicializa com zeros/nulos para garantir que todas as colunas sejam preenchidas
        $notas_a_salvar_por_trimestre = [
            1 => ['mac' => 0, 'npp' => 0, 'npt' => 0, 'mt' => 0, 'mfd' => 0, 'mf' => 0, 'neo' => 0, 'nee' => 0, 'mec' => 0, 'exame' => null, 'nota_final' => null],
            2 => ['mac' => 0, 'npp' => 0, 'npt' => 0, 'mt' => 0, 'mfd' => 0, 'mf' => 0, 'neo' => 0, 'nee' => 0, 'mec' => 0, 'exame' => null, 'nota_final' => null],
            3 => ['mac' => 0, 'npp' => 0, 'npt' => 0, 'mt' => 0, 'mfd' => 0, 'mf' => 0, 'neo' => 0, 'nee' => 0, 'mec' => 0, 'exame' => null, 'nota_final' => null]
        ];

        // Buscar TODAS as notas existentes para o aluno em todos os trimestres
        $stmt_busca_todas->bind_param("iiis", $aluno_id, $disciplina_id, $turma_id, $ano_lectivo);
        $stmt_busca_todas->execute();
        $resultado_busca_todas = $stmt_busca_todas->get_result();
        while ($row = $resultado_busca_todas->fetch_assoc()) {
            $t_existente = (int)$row['trimestre'];
            // Preenche com os dados existentes, garantindo que 'mac' e 'npp' são lidos se existirem
            $notas_a_salvar_por_trimestre[$t_existente]['mac'] = $row['mac'] ?? 0;
            $notas_a_salvar_por_trimestre[$t_existente]['npp'] = $row['npp'] ?? 0;
            $notas_a_salvar_por_trimestre[$t_existente]['npt'] = $row['npt'] ?? 0;
            $notas_a_salvar_por_trimestre[$t_existente]['mt']  = $row['mt']  ?? 0; // 'mt' aqui é a MT do trimestre específico (mt1, mt2, mt3)

            // Copia as notas finais que podem estar salvas no trimestre 3, se existirem
            if ($t_existente == 3) {
                $notas_a_salvar_por_trimestre[3]['mfd'] = $row['mfd'] ?? 0;
                $notas_a_salvar_por_trimestre[3]['mf']  = $row['mf']  ?? 0;
                $notas_a_salvar_por_trimestre[3]['neo'] = $row['neo'] ?? 0;
                $notas_a_salvar_por_trimestre[3]['nee'] = $row['nee'] ?? 0;
                $notas_a_salvar_por_trimestre[3]['mec'] = $row['mec'] ?? 0;
                $notas_a_salvar_por_trimestre[3]['exame'] = $row['exame'] ?? null;
                $notas_a_salvar_por_trimestre[3]['nota_final'] = $row['nota_final'] ?? null;
            }
        }

        // Sobrescrever as notas do trimestre SELECIONADO com os valores do POST
        $notas_a_salvar_por_trimestre[$trimestre_selecionado]['mac'] = floatval($_POST["mac_{$trimestre_selecionado}_{$aluno_id}"] ?? 0);
        $notas_a_salvar_por_trimestre[$trimestre_selecionado]['npp'] = floatval($_POST["npp_{$trimestre_selecionado}_{$aluno_id}"] ?? 0);
        $notas_a_salvar_por_trimestre[$trimestre_selecionado]['npt'] = floatval($_POST["npt_{$trimestre_selecionado}_{$aluno_id}"] ?? 0);
        
        // As notas MT1, MT2, MT3, MFD, MF, NEO, NEE, MEC já vêm calculadas do frontend
        // e são salvas no POST com seus nomes específicos (ex: mt1_ALUNOID)
        $notas_a_salvar_por_trimestre[1]['mt'] = floatval($_POST["mt1_{$aluno_id}"] ?? $notas_a_salvar_por_trimestre[1]['mt']);
        $notas_a_salvar_por_trimestre[2]['mt'] = floatval($_POST["mt2_{$aluno_id}"] ?? $notas_a_salvar_por_trimestre[2]['mt']);
        $notas_a_salvar_por_trimestre[3]['mt'] = floatval($_POST["mt3_{$aluno_id}"] ?? $notas_a_salvar_por_trimestre[3]['mt']);

        // Notas finais (sempre relacionadas ao Trimestre 3 para serem consistentes)
        $notas_a_salvar_por_trimestre[3]['mfd'] = floatval($_POST["mfd_{$aluno_id}"] ?? 0);
        $notas_a_salvar_por_trimestre[3]['mf']  = floatval($_POST["mf_{$aluno_id}"] ?? 0);
        $notas_a_salvar_por_trimestre[3]['neo'] = floatval($_POST["neo_{$aluno_id}"] ?? 0);
        $notas_a_salvar_por_trimestre[3]['nee'] = floatval($_POST["nee_{$aluno_id}"] ?? 0);
        $notas_a_salvar_por_trimestre[3]['mec'] = floatval($_POST["mec_{$aluno_id}"] ?? 0);


        // Validação no lado do servidor para as notas editáveis no trimestre atual
        $current_mac = $notas_a_salvar_por_trimestre[$trimestre_selecionado]['mac'];
        $current_npp = $notas_a_salvar_por_trimestre[$trimestre_selecionado]['npp'];
        $current_npt = $notas_a_salvar_por_trimestre[$trimestre_selecionado]['npt'];
        $current_neo = $notas_a_salvar_por_trimestre[3]['neo']; // NEO/NEE são sempre do 3º trimestre
        $current_nee = $notas_a_salvar_por_trimestre[3]['nee'];

        if (($current_mac > 20 || $current_mac < 0) ||
            ($current_npp > 20 || $current_npp < 0) ||
            ($current_npt > 20 || $current_npt < 0) ||
            ($current_neo > 20 || $current_neo < 0) ||
            ($current_nee > 20 || $current_nee < 0)
        ) {
             throw new Exception("Erro: As notas do aluno ID {$aluno_id} devem estar entre 0 e 20. Verifique MAC, NPP, NPT, NEO e NEE.");
        }

        // Executar o statement de UPSERT para o trimestre ATUAL
        // O REPLACE INTO irá inserir uma nova linha se não existir, ou substituir se a combinação de chaves existir
        // A combinação de chaves deve ser (aluno_id, disciplina_id, turma_id, ano_lectivo, trimestre)
        $stmt_upsert->bind_param(
            "iiisiddddddddddddddd",
            $aluno_id,
            $turma_id,
            $disciplina_id,
            $ano_lectivo,
            $trimestre_selecionado, // Trimestre atual
            $classe,
            $notas_a_salvar_por_trimestre[$trimestre_selecionado]['mac'],
            $notas_a_salvar_por_trimestre[$trimestre_selecionado]['npp'],
            $notas_a_salvar_por_trimestre[$trimestre_selecionado]['npt'],
            $notas_a_salvar_por_trimestre[$trimestre_selecionado]['mt'], // MT do trimestre atual
            $notas_a_salvar_por_trimestre[1]['mt'], // MT1
            $notas_a_salvar_por_trimestre[2]['mt'], // MT2
            $notas_a_salvar_por_trimestre[3]['mt'], // MT3
            $notas_a_salvar_por_trimestre[3]['mfd'],
            $notas_a_salvar_por_trimestre[3]['mf'],
            $notas_a_salvar_por_trimestre[3]['neo'],
            $notas_a_salvar_por_trimestre[3]['nee'],
            $notas_a_salvar_por_trimestre[3]['mec'],
            $notas_a_salvar_por_trimestre[3]['exame'],
            $notas_a_salvar_por_trimestre[3]['nota_final']
        );

        if (!$stmt_upsert->execute()) {
            throw new Exception("Erro ao salvar notas do aluno ID {$aluno_id} para o trimestre {$trimestre_selecionado}: " . $stmt_upsert->error);
        }
    }
    
    $stmt_busca_todas->close();
    $stmt_upsert->close();
    $conn->commit();
    
    // Redirecionar de volta para a página da pauta com uma mensagem de sucesso
    $url = "mini_pauta_professor.php?turma_id={$turma_id}&disciplina_id={$disciplina_id}&trimestre={$trimestre_selecionado}&sucesso=1";
    header("Location: {$url}");
    exit;

} catch (Exception $e) {
    $conn->rollback();
    
    // Em caso de erro, redirecionar com uma mensagem de erro
    $url = "mini_pauta_professor.php?turma_id={$turma_id}&disciplina_id={$disciplina_id}&trimestre={$trimestre_selecionado}&erro=" . urlencode($e->getMessage());
    header("Location: {$url}");
    exit;
}

$conn->close();
?>
